(**************************************************************
 *         CS101: Homework 4 --- Types                        *
 **************************************************************)

(*
 * Introduce a type alias to make it more obvious which strings are meant
 * to be interpreted as variable names or locations
 *)
module VarMap = Map.Make(String)
type var = VarMap.key

(*
 * Locations to use for evaluating references.
 * We want to keep the type of locations abstract.
 *)
module type LocationsSig = sig
   type t
   type 'a state
   val fresh : unit -> t
   val empty_state : 'a state
   val add : 'a state -> t -> 'a -> 'a state (* replaces any existing mappings *)
   val find : 'a state -> t -> 'a (* May raise Not_found *)
end

module Location : LocationsSig = struct
   type t = int

   module MapBase = struct
      type t = int
      let compare = Pervasives.compare
   end

   module LocMap = Map.Make(MapBase)

   type 'a state = 'a LocMap.t
   
   let fresh =
      let count = ref 0 in
       fun () ->
         count := !count + 1;
         !count

   let empty_state = LocMap.empty
   let add map loc v = LocMap.add loc v map
   let find map loc = LocMap.find loc map
end

type location = Location.t

(*
 * Lambda-calculus types:
 *   t := int
 *     |  t -> t
 *)
type lc_type =
   TyInt
 | TyFun of lc_type * lc_type
 | TyUnit
 | TyRef of lc_type

(*
 * Lambda-calculus binary operations: +, -, *
 *)
type lc_binop =
   OpPlus
 | OpMinus
 | OpTimes

(*
 * Lambda-calculus expressions:
 *   e := n
 *     |  x
 *     |  e op e
 *     |  e e
 *     | lambda x: t. e
 *)
type lc_exp =
   ExpInt of int
 | ExpVar of var
 | ExpBinop of lc_binop * lc_exp * lc_exp
 | ExpApply of lc_exp * lc_exp
 | ExpLambda of var * lc_type * lc_exp
 | ExpUnit
 | ExpRef of lc_exp
 | ExpDeref of lc_exp
 | ExpAssign of lc_exp * lc_exp

type lc_val =
   ValInt of int
 | ValUnit of unit
 | ValClosure of value_environment * var * lc_exp
 | ValRef of location
 
and value_environment = lc_val VarMap.t

(*
 * The result of an LC expression evaluation is a state and a value
 *)
type lc_full_value = {
   refs : lc_val Location.state;
   value : lc_val
}

(*
 * Type errors:
 *   - "Expression e has type t1, but is used with type t2"
 *   - "Expression e of non-functional type t is used in a function position"
 *   - "Variable v is not bound"
 *)
type lc_error =
      ErrTypeMismatch of lc_exp * lc_type * lc_type
   |  ErrTypeFunApp of lc_exp * lc_type
   |  ErrUnknownVariable of var

exception LcError of lc_error
exception InternalError of string
